#!/usr/bin/env python3
"""
scripts/compute_anomalies.py

Compute anomaly indicators from path‑integral samples.
Reads HDF5 samples, computes anomaly_indicator for each sample,
and saves the results as a NumPy array.
"""

import argparse
import os
import numpy as np
import h5py
from analysis.observables import load_samples, anomaly_indicator

def main():
    parser = argparse.ArgumentParser(
        description="Compute anomaly indicators from path‑integral samples"
    )
    parser.add_argument(
        "--samples", "-s",
        type=str,
        default="results/samples.h5",
        help="Path to input HDF5 file with dataset 'samples'"
    )
    parser.add_argument(
        "--output", "-o",
        type=str,
        default="results/anomalies.npy",
        help="Path to output NumPy file for anomaly indicators"
    )
    args = parser.parse_args()

    # Load field samples
    samples = load_samples(args.samples)

    # Compute anomaly indicators for each sample
    anomalies = []
    for field in samples:
        val = anomaly_indicator(field)
        anomalies.append(val if val is not None else np.nan)
    anomalies = np.array(anomalies, dtype=float)

    # Ensure output directory exists
    out_dir = os.path.dirname(args.output)
    if out_dir and not os.path.exists(out_dir):
        os.makedirs(out_dir)

    # Save anomalies array
    np.save(args.output, anomalies)
    print(f"Saved anomaly indicators (shape={anomalies.shape}) to {args.output}")

if __name__ == "__main__":
    main()
